/**
 * \file: mspin_lm_framebuffer_adapter.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MySpin LayerManager Framebuffer Adapter using APX
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES thilobjoern.fickel@bosch-softtec.com
 *
 * \copyright: (c) 2015 Bosch SoftTec
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_lm_framebuffer_adapter.h"
#include "mspin_lm_adapter_internal.h"
#include "mspin_logging.h"

#include <string.h>

//Enable MSPIN_LM_USE_DOUBLE_BUFFERING as compile flag. If not possible enable it here:
// MSPIN_LM_USE_DOUBLE_BUFFERING enables double buffering instead of using single buffering
//#define MSPIN_LM_USE_DOUBLE_BUFFERING

struct mspin_framebuffer_context
{
    struct apx apxContext;
    struct apx_buffer* apxBuffer;
};

void mspin_framebuffer_initGlobalHandler(void* data, struct wl_registry* registry, uint32_t name,
        const char* interface, uint32_t version)
{
    mspin_wl_context_t *pWLCtx = (mspin_wl_context_t*)data;
    if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(data=%p, registry=%p, name=%d, interface='%s', version=%d) ERROR: Wayland context is NULL",
                __FUNCTION__, data, registry, name, interface ? interface : "n/a", version);
        return;
    }

    int result = apx_global_handler(&pWLCtx->pFramebufferContext->apxContext, registry, name, interface, version);
    if (result < 0)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(data=%p, registry=%p, name=%d, interface='%s', version=%d) ERROR: apx_global_handler() failed with code=%d",
                __FUNCTION__, data, registry, name, interface ? interface : "n/a", version, result);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug,
                "%s(data=%p, registry=%p, name=%d, interface='%s', version=%d) apx_global_handler() succeeded",
                __FUNCTION__, data, registry, name, interface ? interface : "n/a", version);
    }
}

int mspin_framebuffer_init(mspin_wl_context_t *pWLCtx)
{
    if (pWLCtx)
    {
        pWLCtx->pFramebufferContext = malloc(sizeof(mspin_framebuffer_context));
        if (pWLCtx->pFramebufferContext)
        {
            memset(pWLCtx->pFramebufferContext, 0, sizeof(mspin_framebuffer_context));
            return MSPIN_LM_SUCCESS;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p) ERROR: Failed to allocate memory for apx context",
                    __FUNCTION__, pWLCtx);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s(wlCtx=%p) ERROR: pWLCtx is NULL", __FUNCTION__, pWLCtx);
    }

    return MSPIN_LM_FAIL;
}


int mspin_framebuffer_createImage(mspin_wl_context_t *pWLCtx, int width, int height)
{
    unsigned int stride;
    if (pWLCtx && pWLCtx->pFramebufferContext)
    {
        int result = apx_init(&pWLCtx->pFramebufferContext->apxContext);
        if (0 > result)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_init() failed with code=%d",
                    __FUNCTION__, pWLCtx, width, height, result);
            return MSPIN_LM_FAIL;
        }

#ifdef MSPIN_LM_USE_DOUBLE_BUFFERING
#ifdef MSPIN_LM_USE_OLD_CREATE_BUFFER_API
        pWLCtx->pFramebufferContext->apxBuffer = apx_buffer_create(&pWLCtx->pFramebufferContext->apxContext,
                width, height, pWLCtx->surface_format);
#else
        pWLCtx->pFramebufferContext->apxBuffer = apx_buffer_create(&pWLCtx->pFramebufferContext->apxContext,
                width, height, (apxPixelFormat)pWLCtx->surface_format);
#endif //MSPIN_LM_USE_OLD_CREATE_BUFFER_API

        if (!pWLCtx->pFramebufferContext->apxBuffer)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_buffer_create() failed",
                    __FUNCTION__, pWLCtx, width, height);
            return MSPIN_LM_FAIL;
        }
#else
#ifdef MSPIN_LM_USE_OLD_CREATE_BUFFER_API
        pWLCtx->pFramebufferContext->apxBuffer = apx_buffer_create_unbuffered(&pWLCtx->pFramebufferContext->apxContext,
                width, height, pWLCtx->surface_format);
#else
        pWLCtx->pFramebufferContext->apxBuffer = apx_buffer_create_unbuffered(&pWLCtx->pFramebufferContext->apxContext,
                width, height, (apxPixelFormat)pWLCtx->surface_format);
#endif //MSPIN_LM_USE_OLD_CREATE_BUFFER_API

        if (!pWLCtx->pFramebufferContext->apxBuffer)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_buffer_create_unbuffered() failed",
                    __FUNCTION__, pWLCtx, width, height);
            return MSPIN_LM_FAIL;
        }
#endif //MSPIN_LM_USE_DOUBLE_BUFFERING


        if (apx_buffer_get_stride(pWLCtx->pFramebufferContext->apxBuffer, &stride) < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_buffer_get_stride() failed",
                    __FUNCTION__, pWLCtx, width, height);
            mspin_framebuffer_destroyImage(pWLCtx);
            return MSPIN_LM_FAIL;
        }

        pWLCtx->stride = (t_ilm_int)stride;

        if (apx_buffer_map(pWLCtx->pFramebufferContext->apxBuffer, (void **) &(pWLCtx->p_image[0])) < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_buffer_map() failed",
                    __FUNCTION__, pWLCtx, width, height);
            mspin_framebuffer_destroyImage(pWLCtx);
            return MSPIN_LM_FAIL;
        }
    }
    else if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: pWLCtx is NULL",
                __FUNCTION__, pWLCtx, width, height);
        return MSPIN_LM_FAIL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: p_lm_apx_context is NULL",
                __FUNCTION__, pWLCtx, width, height);
        return MSPIN_LM_FAIL;
    }

    pWLCtx->initialized = TRUE;

#ifdef MSPIN_LM_USE_DOUBLE_BUFFERING
    mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p, w=%d, h=%d) Image buffers created for double buffering",
            __FUNCTION__, pWLCtx, width, height);
#else
    mspin_log_printLn(eMspinVerbosityInfo, "%s(ctx=%p, w=%d, h=%d) Image buffers created for single buffering",
            __FUNCTION__, pWLCtx, width, height);
#endif //MSPIN_LM_USE_DOUBLE_BUFFERING

    return MSPIN_LM_SUCCESS;
}


void mspin_framebuffer_destroyImage(mspin_wl_context_t *pWLCtx)
{
    if (pWLCtx && pWLCtx->pFramebufferContext)
    {
        pWLCtx->initialized = FALSE;

        if (pWLCtx->pFramebufferContext->apxBuffer)
        {
            apx_buffer_destroy(pWLCtx->pFramebufferContext->apxBuffer, NULL/*sink->wl_display*/);
            pWLCtx->pFramebufferContext->apxBuffer = NULL;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s(ctx=%p, w=%d, h=%d) WARNING: apx_buf is NULL",
                    __FUNCTION__, pWLCtx);
        }

        if (apx_deinit(&pWLCtx->pFramebufferContext->apxContext) < 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: apx_deinit() failed",
                    __FUNCTION__, pWLCtx);
        }
    }
    else if (!pWLCtx)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: ctx is NULL",
                __FUNCTION__, pWLCtx);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(ctx=%p, w=%d, h=%d) ERROR: p_lm_apx_context is NULL",
                __FUNCTION__, pWLCtx);
    }
}


void mspin_framebuffer_drawImage(mspin_wl_context_t *pWLCtx, int width, int height)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(wlCtx=%p)", __FUNCTION__, pWLCtx);

    if (pWLCtx)
    {
        if (pWLCtx->initialized)
        {
            if (pWLCtx->pFramebufferContext)
            {
                if (apx_buffer_commit(&pWLCtx->pFramebufferContext->apxContext,
                        pWLCtx->pFramebufferContext->apxBuffer, pWLCtx->p_wl_surface, 0, 0,
                        width, height, pWLCtx->p_wl_display) < 0)
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p) ERROR: apx_buffer_commit() failed",
                            __FUNCTION__, pWLCtx);
                    return;
                }

                if (apx_buffer_map(pWLCtx->pFramebufferContext->apxBuffer, (void **)&(pWLCtx->p_image)) < 0)
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p) ERROR: apx_buffer_map() failed",
                            __FUNCTION__, pWLCtx);
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p) ERROR: framebuffer context is NULL",
                        __FUNCTION__, pWLCtx);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s(wlCtx=%p) WARNING: wlCtx isn't initialized",
                    __FUNCTION__, pWLCtx);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(wlCtx=%p) ERROR: wlCtx is NULL",
                __FUNCTION__, pWLCtx);
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(wlCtx=%p) -> end", __FUNCTION__, pWLCtx);
}
